/*:
 * @target MZ
 * @plugindesc [MEG] HS_OptionsOverlayを閉じた直後に指定コモンイベントを1回実行（UI再描画用） v1.1.0
 * @author
 *
 * @param commonEventId
 * @text UI再描画コモンイベントID
 * @type common_event
 * @default 900
 *
 * @param delayFrames
 * @text 実行までの遅延フレーム数
 * @type number
 * @min 0
 * @default 2
 * @desc オプションを閉じた後、何フレーム待ってからCEを予約するか（2〜3推奨）
 *
 * @param debugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 *
 * @help
 * HS_OptionsOverlay は Scene_Options に遷移せず、シーン上に Options ウィンドウを重ねる方式です。
 * そのため Scene_Options のフックでは動きません。
 *
 * 本プラグインは scene._hsOptionsWin の「開→閉」を検知し、
 * 指定のコモンイベントを「1回だけ」予約します。
 *
 * 注意：
 * - UI再描画CEは短く（Wait/メッセージ等は避ける）してください。
 * - プラグインは HS_OptionsOverlay / HS_OptionsOverlay_InputLock より下に配置してください。
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "MEG_OptionsOverlay_CloseRefreshCE";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const CE_ID = Number(params.commonEventId || 900);
  const DELAY_FRAMES = Math.max(0, Number(params.delayFrames || 2));
  const DEBUG = String(params.debugLog) === "true";

  function log(...a) {
    if (DEBUG) console.log(`[${PLUGIN_NAME}]`, ...a);
  }

  function isCommonEventReservedSafe() {
    return !!($gameTemp && $gameTemp.isCommonEventReserved && $gameTemp.isCommonEventReserved());
  }

  const _SceneBase_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _SceneBase_update.call(this);

    // HS_OptionsOverlay が保持しているウィンドウ参照（開いてる間は truthy）
    const isOpen = !!this._hsOptionsWin;

    // 前フレームの状態
    const wasOpen = !!this._megWasOptionsOverlayOpen;
    this._megWasOptionsOverlayOpen = isOpen;

    // 「開いていた→閉じた」を検知
    if (wasOpen && !isOpen) {
      this._megOverlayCloseCountdown = DELAY_FRAMES;
      log("Overlay closed. countdown =", this._megOverlayCloseCountdown);
      // 0フレーム指定なら次の処理で即実行
    }

    // 遅延カウント
    const cd = this._megOverlayCloseCountdown || 0;
    if (cd > 0) {
      this._megOverlayCloseCountdown = cd - 1;
      return;
    }

    // 0になったら予約（ただし実行できない状況なら1フレーム延期）
    if (this._megOverlayCloseCountdown === 0) {
      // タイトル等で $gameMap が無い場合は何もしない
      if (!$gameTemp || !$gameMap) {
        this._megOverlayCloseCountdown = -1;
        return;
      }

      // すでにイベント実行中/CE予約中なら延期
      if (($gameMap.isEventRunning && $gameMap.isEventRunning()) || isCommonEventReservedSafe()) {
        this._megOverlayCloseCountdown = 1;
        return;
      }

      // 実行
      this._megOverlayCloseCountdown = -1; // 二重実行防止
      if (CE_ID > 0) {
        $gameTemp.reserveCommonEvent(CE_ID);
        log("reserveCommonEvent:", CE_ID);
      }
    }
  };
})();
